<?php
defined('ABSPATH') || exit;

// Include the connectivity functions
require_once plugin_dir_path(dirname(__FILE__)) . 'includes/connectivity.php';

// Remove all admin notices from this page to prevent UI breakage
function wpfort_remove_admin_notices() {
    if (isset($_GET['page']) && $_GET['page'] === 'wpsec-dashboard') {
        // Remove standard WordPress admin notices
        remove_all_actions('admin_notices');
        remove_all_actions('all_admin_notices');
        remove_all_actions('user_admin_notices');
        remove_all_actions('network_admin_notices');
        
        // Remove Elementor admin notices specifically
        if (class_exists('\Elementor\Plugin')) {
            remove_action('admin_notices', [\Elementor\Plugin::$instance->admin, 'admin_notices']);
        }
        
        // Buffer the output to remove any notices that got through
        add_action('admin_head', function() {
            ob_start();
        }, -999);
        
        add_action('admin_footer', function() {
            $content = ob_get_clean();
            
            // Strip out any notices that might have been output
            $content = preg_replace('/<div[^>]*class="[^"]*(?:notice|error|updated|update-nag)[^"]*".*?<\/div>/s', '', $content);
            $content = preg_replace('/<div[^>]*class="[^"]*(?:elementor|e-notice)[^"]*".*?<\/div>/s', '', $content);
            $content = preg_replace('/<div[^>]*id="[^"]*(?:elementor)[^"]*".*?<\/div>/s', '', $content);
            $content = preg_replace('/<div[^>]*class="[^"]*(?:dialog)[^"]*".*?<\/div>/s', '', $content);
            
            echo wp_kses_post($content);
        }, 999);
        
        // Also add CSS as a final fallback
        add_action('admin_head', function() {
            $css_styles = '
                .notice, .error, .updated, .update-nag, .notice-error, .notice-warning, .notice-success, .notice-info,
                .elementor-admin-dialog, .elementor-dialog-widget, .dialog-widget, .dialog-message, .dialog-type-lightbox,
                div[id^="elementor"], div[class^="elementor"], .e-notice, .e-admin-top-bar {
                    display: none !important;
                }
                
                .wpfort-dashboard {
                    z-index: 9999 !important;
                    position: relative;
                }';
            echo '<style>' . esc_html($css_styles) . '
                
                /* Make our dashboard take over the page */
                #wpbody-content > :not(.wpfort-dashboard) {
                    display: none !important;
                }
            </style>';
        });
        
        // Dequeue scripts that might cause notices
        add_action('admin_enqueue_scripts', function() {
            global $wp_scripts, $wp_styles;
            
            // Dequeue all scripts and styles that contain 'elementor'
            foreach ($wp_scripts->registered as $handle => $script) {
                if (strpos($handle, 'elementor') !== false) {
                    wp_dequeue_script($handle);
                }
            }
            
            foreach ($wp_styles->registered as $handle => $style) {
                if (strpos($handle, 'elementor') !== false) {
                    wp_dequeue_style($handle);
                }
            }
        }, 999);
    }
}
add_action('in_admin_header', 'wpfort_remove_admin_notices', 5); // Very early priority

// Handle refresh request (clear cache and force new API call)
if (isset($_GET['refresh']) && $_GET['refresh'] === '1') {
    // Delete the transient to force a fresh API call
    delete_transient('wpsec_site_connectivity');
    
    // Log the refresh action
    wpsec_debug_log('WPFort: Manual refresh of connectivity data requested', 'info');
}

// Get connectivity data
$connectivity_data = wpsec_get_site_connectivity();
$overall_status = wpsec_get_overall_connectivity($connectivity_data);
$ping_status = wpsec_get_ping_status($connectivity_data);
$wpcore_layer = wpsec_check_layer_status('wpcore_layer', $connectivity_data);
$filesystem_layer = wpsec_check_layer_status('filesystem_layer', $connectivity_data);
$application_layer = wpsec_check_layer_status('application_layer', $connectivity_data);
$network_layer = wpsec_check_layer_status('network_layer', $connectivity_data);

// Handle authorization code generation
if (isset($_POST['wpsec_action']) && $_POST['wpsec_action'] === 'generate_auth_code') {
    if (check_admin_referer('wpsec_generate_auth_code', 'wpsec_auth_nonce')) {
        // Generate a random 6-character alphanumeric code
        $new_auth_code = wp_generate_password(6, false, false);
        
        // Convert to uppercase to ensure only capital letters and numbers (no lowercase)
        $new_auth_code = strtoupper($new_auth_code);
        
        // Save the new auth code
        update_option('wpsec_site_auth_code', $new_auth_code);
        
        // Redirect to avoid form resubmission
        echo '<script>window.location.href = "' . esc_url(admin_url('admin.php?page=wpsec-dashboard&code_generated=1')) . '";</script>';
        exit;
    }
}

// Get the current auth code
$auth_code = get_option('wpsec_site_auth_code', '');

// CSS styles for the dashboard
wp_enqueue_style('dashicons');
?>

<style>
.wpfort-dashboard {
    max-width: 1200px;
    margin: 0 auto;
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
}

.wpfort-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 20px;
    padding: 20px;
    border-radius: 8px;
    background: linear-gradient(90deg, #0e2764 0%, #1e3c7b 100%);
    color: white;
}

.wpfort-header h1 {
    margin: 0;
    color: white;
    font-size: 24px;
    font-weight: 600;
}

.wpfort-header-logo {
    display: flex;
    align-items: center;
}

.wpfort-header-logo img {
    width: 40px;
    height: 40px;
    margin-right: 10px;
}

.wpfort-cards-container {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 20px;
    margin-bottom: 20px;
}

.wpfort-card {
    background-color: white;
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    padding: 20px;
    position: relative;
}

.wpfort-card h2 {
    margin-top: 0;
    font-size: 18px;
    color: #0e2764;
    display: flex;
    align-items: center;
    border-bottom: 1px solid #f0f0f0;
    padding-bottom: 10px;
    margin-bottom: 15px;
}

.wpfort-card h2 i {
    margin-right: 8px;
}

.wpfort-status {
    display: flex;
    align-items: center;
    margin: 10px 0;
}

.wpfort-status-indicator {
    width: 12px;
    height: 12px;
    border-radius: 50%;
    margin-right: 10px;
}

.wpfort-status-indicator.connected {
    background-color: #46b450;
}

.wpfort-status-indicator.disconnected {
    background-color: #dc3232;
}

.wpfort-status-indicator.warning {
    background-color: #ffb900;
}

.wpfort-btn {
    display: inline-block;
    padding: 10px 20px;
    border-radius: 4px;
    text-decoration: none;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.wpfort-resync-button {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 5px;
    font-weight: 600;
    margin-left: 10px;
    background-color: #0e2764;
    border-color: #0e2764;
}

.wpfort-resync-button:hover,
.wpfort-resync-button:focus {
    background-color: #172f6d;
    border-color: #172f6d;
}

.wpfort-resync-button .dashicons {
    font-size: 16px;
    width: 16px;
    height: 16px;
    line-height: 16px;
    margin: 0;
    vertical-align: middle;
}

.wpfort-button-group {
    margin-top: 15px;
}

.wpfort-sentinel-icon {
    width: 20px;
    height: 20px;
    margin: 0 5px 0 5px;
    vertical-align: middle;
}

.sentinel-status {
    margin: 15px 0;
    padding: 8px 10px;
    background-color: #f0f7ff;
    border-radius: 4px;
    border-left: 3px solid #0e2764;
}

.wpfort-btn-primary {
    background-color: #0e2764;
    color: white;
    border: none;
}

.wpfort-btn-primary:hover {
    background-color: #1e3c7b;
    color: white;
}

.wpfort-btn-secondary {
    background-color: #f8f9fa;
    color: #0e2764;
    border: 1px solid #0e2764;
}

.wpfort-btn-connected {
    background-color: #f1f5ff;
    color: #0e2764;
    border: 2px solid #0e2764;
    font-weight: 600;
    padding: 12px 25px;
    border-radius: 4px;
    transition: all 0.3s ease;
    box-shadow: 0 3px 6px rgba(14, 39, 100, 0.1);
}

.wpfort-btn-connected:hover {
    background-color: #e1ebff;
    box-shadow: 0 5px 10px rgba(14, 39, 100, 0.2);
}

.wpfort-btn-secondary:hover {
    background-color: #efefef;
}

.wpfort-alert {
    padding: 15px;
    border-radius: 4px;
    margin: 15px 0;
    border-left: 4px solid #ddd;
}

.wpfort-alert-info {
    background-color: #f8f8f8;
    border-color: #bce8f1;
}

.wpfort-layer {
    margin-bottom: 15px;
    padding: 12px;
    border-radius: 6px;
    background-color: #f9f9f9;
    border-left: 4px solid #ddd;
    cursor: pointer;
    transition: all 0.2s ease;
}

.wpfort-layer:hover {
    background-color: #f3f3f3;
}

.wpfort-layer.active {
    border-left-color: #46b450;
}

.wpfort-layer.inactive {
    border-left-color: #dc3232;
}

.wpfort-layer h3 {
    margin: 0 0 5px 0;
    font-size: 16px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.wpfort-layer p {
    margin: 5px 0;
    font-size: 13px;
    color: #666;
}

.wpfort-code-display {
    background: #f7f7f7;
    border: 1px solid #ddd;
    padding: 15px;
    border-radius: 4px;
    font-family: monospace;
    font-size: 16px;
    margin: 10px 0;
    text-align: center;
}

.wpfort-footer {
    margin-top: 30px;
    padding: 15px;
    background-color: #f9f9f9;
    border-radius: 8px;
    font-size: 12px;
    color: #666;
    text-align: center;
}

.wpfort-footer a {
    color: #0e2764;
    text-decoration: none;
}

.wpfort-notice {
    padding: 10px 15px;
    margin: 15px 0;
    background-color: #fafafa;
    border-left: 4px solid #0e2764;
    font-size: 13px;
    color: #555;
}

/* Admin notices container styling */
.wpfort-admin-notices {
    margin-bottom: 20px;
    position: relative;
    z-index: 1;
}

/* Connection Overlay Styles */
.wpfort-connection-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: rgba(0, 0, 0, 0.7);
    z-index: 99999;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 20px;
}

.wpfort-connection-box {
    background-color: white;
    border-radius: 8px;
    max-width: 700px;
    width: 100%;
    padding: 40px;
    text-align: center;
    box-shadow: 0 5px 20px rgba(0, 0, 0, 0.2);
}

.wpfort-connection-logo {
    margin-bottom: 30px;
}

.wpfort-shield-icon {
    width: 80px;
    height: auto;
}

.wpfort-connection-title {
    font-size: 26px;
    margin-bottom: 25px;
    color: #0e2764;
    line-height: 1.4;
}

.wpfort-connection-tagline {
    font-size: 18px;
    margin-bottom: 30px;
    color: #444;
    line-height: 1.5;
    max-width: 90%;
    margin-left: auto;
    margin-right: auto;
}

.wpfort-connection-cta {
    margin: 25px 0;
}

.wpfort-connection-feature-text {
    font-size: 16px;
    font-weight: 500;
    color: #0e2764;
    margin: 20px 0;
}

.wpfort-connection-secondary-action {
    margin: 25px 0;
}

.wpfort-connection-disclaimer {
    font-size: 13px;
    color: #666;
    margin-top: 25px;
    line-height: 1.5;
    max-width: 90%;
    margin-left: auto;
    margin-right: auto;
}

/* Pricing Table Styles */
.wpfort-pricing-section {
    margin-top: 40px;
    background: #fff;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
}

.wpfort-section-title {
    font-size: 24px;
    margin-bottom: 20px;
    color: #0e2764;
    text-align: center;
}

.wpfort-pricing-table {
    overflow-x: auto;
    margin-bottom: 20px;
}

.wpfort-pricing-table table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.wpfort-pricing-table th,
.wpfort-pricing-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
    vertical-align: top;
}

.wpfort-pricing-table thead th {
    background-color: #f8f9fa;
    font-weight: 600;
    color: #333;
    border-top: 1px solid #e9ecef;
    vertical-align: bottom;
    position: sticky;
    top: 0;
}

.wpfort-pricing-table th.highlight,
.wpfort-pricing-table td.highlight {
    background-color: #f0f7ff;
    border-left: 2px solid #0e2764;
    border-right: 2px solid #0e2764;
}

.wpfort-pricing-table tbody tr:hover {
    background-color: #f8f9fa;
}

.wpfort-pricing-table .feature-name {
    font-weight: 600;
    min-width: 120px;
}

.wpfort-pricing-table .plan-header {
    font-weight: 700;
    font-size: 16px;
    color: #0e2764;
    margin-bottom: 5px;
}

.wpfort-pricing-table .plan-subheader {
    font-size: 12px;
    color: #666;
}

.wpfort-pricing-table .check {
    color: #28a745;
    font-weight: bold;
    margin-right: 5px;
}

.wpfort-pricing-table .cross {
    color: #dc3545;
    font-weight: bold;
    margin-right: 5px;
}

.wpfort-pricing-table .warning {
    color: #ffc107;
    font-weight: bold;
    margin-right: 5px;
}

.wpfort-pricing-table .premium {
    color: #0e2764;
    font-weight: bold;
    margin-right: 5px;
}

.wpfort-pricing-table .danger {
    color: #dc3545;
    margin-right: 5px;
}

.wpfort-upgrade-cta {
    text-align: center;
    margin: 20px 0;
}

.wpfort-btn-large {
    padding: 12px 24px;
    font-size: 16px;
}

/* Responsive adjustments */
@media (max-width: 992px) {
    .wpfort-pricing-table {
        overflow-x: auto;
    }
    
    .wpfort-pricing-table table {
        min-width: 800px;
    }
}

/* Custom notice styling */
.wpfort-admin-notices {
    margin: 20px 0;
}

.wpfort-notice {
    padding: 12px 15px;
    margin: 5px 0 15px;
    border-radius: 4px;
    border-left: 4px solid #0e2764;
    background-color: #f0f6ff;
    box-shadow: 0 1px 3px rgba(0,0,0,0.1);
}

.wpfort-notice.notice-success {
    border-color: #46b450;
    background-color: #ecf7ed;
}

/* Make sure our dashboard container has proper z-index */
.wpfort-dashboard {
    position: relative;
    z-index: 0;
}

/* Fix for WordPress notices */
.wrap.wpfort-dashboard > .notice {
    display: none !important; /* Hide notices that appear outside our container */
}

.wpfort-admin-notices .notice {
    margin: 5px 0 15px 0;
}
</style>

<?php
// Create a function to capture admin notices
function wpfort_start_admin_notices_capture() {
    ob_start();
}

function wpfort_end_admin_notices_capture() {
    $notices = ob_get_clean();
    return $notices;
}

// We no longer need to capture admin notices since we're removing them completely
?>

<div class="wrap wpfort-dashboard">
    <!-- Initial Connection Overlay -->
    <?php if (wpsec_needs_initial_connection($connectivity_data)): 
        $site_domain = wp_parse_url(get_site_url(), PHP_URL_HOST);
        $signup_url = 'https://wpfort.ai/signup?from=wpsec&domain=' . urlencode($site_domain);
    ?>
    <div class="wpfort-connection-overlay">
        <div class="wpfort-connection-box">
            <div class="wpfort-connection-logo">
                <img src="<?php echo esc_url(plugin_dir_url(dirname(__FILE__)) . 'assets/images/wpfort-shield.png'); ?>" alt="WPFort Shield" class="wpfort-shield-icon">
            </div>
            <h2 class="wpfort-connection-title">WPFort AI Security Suite Requires Connection to Our Cloud Platform</h2>
            <p class="wpfort-connection-tagline">Activate AI AutoPilot, Real-Time Threat Removal, and gain full live visibility & control over <strong><?php echo esc_html($site_domain); ?></strong></p>
            <div class="wpfort-connection-cta">
                <a href="<?php echo esc_url($signup_url); ?>" target="_blank" id="wpfort-connect-site-btn" class="wpfort-btn wpfort-btn-primary wpfort-btn-large">Connect My Site Now to Transform it into a Fortress</a>
            </div>
            <p class="wpfort-connection-feature-text">WPFort.ai connection activates AI 360 Security, uptime monitoring, enterprise grade firewall and more</p>
            <div class="wpfort-connection-secondary-action" id="wpfort-connected-action" style="display: none;">
                <a href="<?php echo esc_url(add_query_arg('refresh', '1', $_SERVER['REQUEST_URI'])); ?>" class="wpfort-btn wpfort-btn-secondary wpfort-btn-connected">I Connected My Site</a>
            </div>
            <p class="wpfort-connection-disclaimer">By connecting, you'll enable AI-powered security scanning, malware protection, and automatic threat remediation. The WPFort platform communicates with your site via secure, lightweight non-disruptive API calls and helps keep your WordPress installation safe from hackers, malware, and security vulnerabilities. No personal data is shared.</p>
        </div>

        <script type="text/javascript">
            document.addEventListener('DOMContentLoaded', function() {
                var connectButton = document.getElementById('wpfort-connect-site-btn');
                var connectedAction = document.getElementById('wpfort-connected-action');
                
                if (connectButton && connectedAction) {
                    connectButton.addEventListener('click', function() {
                        // Wait a moment to show the secondary action
                        setTimeout(function() {
                            connectedAction.style.display = 'block';
                        }, 500);
                    });
                }
            });
        </script>
    </div>
    <?php endif; ?>
    
    <!-- Admin Notices Container -->
    <div class="wpfort-admin-notices">
        <?php 
        // Show success message after generating a code
        if (isset($_GET['code_generated']) && $_GET['code_generated'] === '1'): ?>
            <div class="wpfort-notice notice-success"><p>Authorization code successfully generated!</p></div>
        <?php endif; ?>
    </div>

    <!-- Header -->
    <div class="wpfort-header">
        <div class="wpfort-header-logo">
            <span style="font-size: 30px; margin-right: 10px;" class="dashicons dashicons-shield"></span>
            <h1>WPFort.ai Security</h1>
        </div>
        <a href="https://www.wpfort.ai/app/dashboard/websites" target="_blank" class="wpfort-btn wpfort-btn-primary">Go to Your WPFort Dashboard</a>
    </div>
    
    <div class="wpfort-cards-container">
        <!-- Site Status Card -->
        <div class="wpfort-card">
            <h2><span class="dashicons dashicons-shield"></span> Site Connection Status</h2>
            
            <div class="wpfort-status">
                <div class="wpfort-status-indicator <?php echo esc_attr(($ping_status['connected']) ? 'connected' : 'disconnected'); ?>"></div>
                <strong>Plugin Connection: </strong>&nbsp;
                <?php echo esc_html(($ping_status['connected']) ? 'Connected' : 'Disconnected'); ?>
            </div>
            
            <?php if ($ping_status['connected']): ?>
                <p>Plugin Version: <?php echo esc_html($ping_status['plugin_version']); ?></p>
            <?php else: ?>
                <p>Unable to establish connection with WPFort servers.</p>
            <?php endif; ?>
            
            <div class="wpfort-status">
                <div class="wpfort-status-indicator <?php echo esc_attr(($overall_status['status'] === 'healthy') ? 'connected' : (($overall_status['status'] === 'warning') ? 'warning' : 'disconnected')); ?>"></div>
                <strong>Monitoring Status: </strong>&nbsp;
                <?php echo esc_html($overall_status['status_text']); ?>
            </div>
            
            <?php if ($overall_status['last_check']): ?>
                <p>Last Check: <?php echo esc_html(wpsec_format_date($overall_status['last_check'])); ?></p>
            <?php endif; ?>
            
            <?php if ($ping_status['connected']): ?>
                <div class="wpfort-status sentinel-status">
                    <div class="wpfort-status-indicator connected"></div>
                    <img src="<?php echo esc_url(plugin_dir_url(dirname(__FILE__)) . 'assets/images/wpfort-sentinel.png'); ?>" alt="Sentinel" class="wpfort-sentinel-icon">
                    <strong>Sentinel AI Agent: </strong>&nbsp;Active
                </div>
            <?php endif; ?>
            
            <div class="wpfort-button-group">
                <p>
                    <a href="https://www.wpfort.ai/app/dashboard/websites" target="_blank" class="button">View in Dashboard</a>
                    <a href="<?php echo esc_url(add_query_arg('refresh', '1', $_SERVER['REQUEST_URI'])); ?>" class="button button-primary wpfort-resync-button"><span class="dashicons dashicons-update"></span> Resync with wpfort.ai</a>
                </p>
            </div>
        </div>
        
        <!-- Security Layers Card -->
        <div class="wpfort-card">
            <h2><span class="dashicons dashicons-admin-site-alt3"></span> Security Layers</h2>
            
            <div class="wpfort-layer <?php echo esc_attr(($wpcore_layer['active']) ? 'active' : 'inactive'); ?>" 
                 onclick="window.open('https://www.wpfort.ai/app/dashboard/websites', '_blank')">
                <h3>
                    WordPress Core Layer
                    <?php if ($wpcore_layer['active']): ?>
                        <span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
                    <?php else: ?>
                        <span class="dashicons dashicons-no-alt" style="color: #dc3232;"></span>
                    <?php endif; ?>
                </h3>
                <p>Status: <?php echo esc_html(($wpcore_layer['active']) ? 'Active' : 'Inactive'); ?></p>
                <?php if ($wpcore_layer['last_updated']): ?>
                    <p>Last Updated: <?php echo esc_html(wpsec_format_date($wpcore_layer['last_updated'])); ?></p>
                <?php endif; ?>
            </div>
            
            <div class="wpfort-layer <?php echo esc_attr(($application_layer['active']) ? 'active' : 'inactive'); ?>" 
                 onclick="window.open('https://www.wpfort.ai/app/dashboard/websites', '_blank')">
                <h3>
                    Vulnerability Hunter
                    <?php if ($application_layer['active']): ?>
                        <span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
                    <?php else: ?>
                        <span class="dashicons dashicons-no-alt" style="color: #dc3232;"></span>
                    <?php endif; ?>
                </h3>
                <p>Status: <?php echo esc_html(($application_layer['active']) ? 'Active' : 'Inactive'); ?></p>
                <?php if ($application_layer['last_updated']): ?>
                    <p>Last Updated: <?php echo esc_html(wpsec_format_date($application_layer['last_updated'])); ?></p>
                <?php endif; ?>
            </div>
            
            <div class="wpfort-layer <?php echo esc_attr(($network_layer['active']) ? 'active' : 'inactive'); ?>" 
                 onclick="window.open('https://www.wpfort.ai/app/dashboard/websites', '_blank')">
                <h3>
                    Firewall Layer
                    <?php if ($network_layer['active']): ?>
                        <span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
                    <?php else: ?>
                        <span class="dashicons dashicons-no-alt" style="color: #dc3232;"></span>
                    <?php endif; ?>
                </h3>
                <p>Status: <?php echo esc_html(($network_layer['active']) ? 'Active' : 'Inactive'); ?></p>
                <?php if ($network_layer['last_updated']): ?>
                    <p>Last Updated: <?php echo esc_html(wpsec_format_date($network_layer['last_updated'])); ?></p>
                <?php endif; ?>
            </div>
            
            <div class="wpfort-layer <?php echo esc_attr(($filesystem_layer['active']) ? 'active' : 'inactive'); ?>" 
                 onclick="window.open('https://www.wpfort.ai/app/dashboard/websites', '_blank')">
                <h3>
                    Malware Hunter
                    <?php if ($filesystem_layer['active']): ?>
                        <span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
                    <?php else: ?>
                        <span class="dashicons dashicons-no-alt" style="color: #dc3232;"></span>
                    <?php endif; ?>
                </h3>
                <p>Status: <?php echo esc_html(($filesystem_layer['active']) ? 'Active' : 'Inactive'); ?></p>
                <?php if ($filesystem_layer['last_updated']): ?>
                    <p>Last Updated: <?php echo esc_html(wpsec_format_date($filesystem_layer['last_updated'])); ?></p>
                <?php endif; ?>
            </div>
        </div>

        <!-- Authorization Code Card -->
        <div class="wpfort-card">
            <h2><span class="dashicons dashicons-lock"></span> Site Authentication</h2>
            <p>This authorization code is used to verify ownership of your site when connecting to the WPFort SaaS platform.</p>
            
            <?php if (!empty($auth_code)): ?>
                <div class="wpfort-code-display"><?php echo esc_html($auth_code); ?></div>
            <?php endif; ?>
            
            <form method="post" action="">
                <?php wp_nonce_field('wpsec_generate_auth_code', 'wpsec_auth_nonce'); ?>
                <input type="hidden" name="wpsec_action" value="generate_auth_code">
                <button type="submit" class="wpfort-btn wpfort-btn-secondary">Generate New Authorization Code</button>
            </form>
            <p class="description">Note: Generating a new code will invalidate any previous code.</p>
        </div>
    </div>
    
    <!-- Main CTA -->
    <div class="wpfort-card" style="text-align: center; margin-top: 20px;">
        <h2 style="justify-content: center;"><span class="dashicons dashicons-chart-area"></span> Complete Security Management</h2>
        <p>Access all security features including firewall, malware scanning, and file integrity monitoring through your WPFort dashboard.</p>
        <p style="margin-top: 20px;">
            <a href="https://www.wpfort.ai/app/dashboard/websites" target="_blank" class="wpfort-btn wpfort-btn-primary" style="padding: 12px 30px; font-size: 16px;">
                Go to Your WPFort Dashboard
            </a>
        </p>
    </div>
    
    <!-- Pricing Comparison Table Section -->
    <p class="wpfort-privacy-notice">
        This plugin connects your WordPress site to the WPFort.ai platform for remote malware scans, security fixes, and threat monitoring. 
        No personal data is shared. <a href="https://www.wpfort.ai/privacy" target="_blank">Learn more in our Privacy Policy</a>.
    </p>
    
    <?php 
    // Check if subscription tier is free or not specified
    if (wpsec_is_free_tier($connectivity_data)): 
    ?>
    <div class="wpfort-pricing-section">
        <h2 class="wpfort-section-title">Upgrade For Complete Protection</h2>
        
        <div class="wpfort-pricing-table">
            <table>
                <thead>
                    <tr>
                        <th>Feature</th>
                        <th class="highlight">
                            <div class="plan-header">WPFort.ai</div>
                            <div class="plan-subheader">AI-Powered, 360 Protection</div>
                        </th>
                        <th>
                            <div class="plan-header">Wordfence</div>
                            <div class="plan-subheader">Traditional</div>
                        </th>
                        <th>
                            <div class="plan-header">MalCare</div>
                            <div class="plan-subheader">Cloud-Based, Manual Response</div>
                        </th>
                        <th>
                            <div class="plan-header">Sucuri</div>
                            <div class="plan-subheader">Cloud CDN + Manual Cleanup</div>
                        </th>
                    </tr>
                </thead>
                <tbody>
                    <!-- Threat Detection -->
                    <tr>
                        <td class="feature-name">Threat Detection</td>
                        <td class="highlight">
                            <span class="check">✓</span> AI & heuristic + signature scan (6 layers)<br>
                            <span class="check">✓</span> Realtime, 24/7 scanning
                        </td>
                        <td>
                            <span class="warning">⚠</span> Signature-based only<br>
                            <span class="cross">✗</span> No AI/heuristics
                        </td>
                        <td>
                            <span class="warning">⚠</span> Limited real-time<br>
                            <span class="check">✓</span> Cloud-based scanning
                        </td>
                        <td>
                            <span class="warning">⚠</span> Signature-based only<br>
                            <span class="cross">✗</span> Slower scan cycles
                        </td>
                    </tr>
                    
                    <!-- Response Time -->
                    <tr>
                        <td class="feature-name">Response Time</td>
                        <td class="highlight">
                            <span class="check">✓</span> Instant threat removal<br>
                            <span class="check">✓</span> One-click cleanup
                        </td>
                        <td>
                            <span class="warning">⚠</span> Manual cleanup<br>
                            <span class="danger">🛑</span> Delays common
                        </td>
                        <td>
                            <span class="warning">⚠</span> Manual cleanup (slow)
                        </td>
                        <td>
                            <span class="warning">⚠</span> Manual + ticket-based cleanup
                        </td>
                    </tr>
                    
                    <!-- Auto-Protection -->
                    <tr>
                        <td class="feature-name">Auto-Protection</td>
                        <td class="highlight">
                            <span class="check">✓</span> Fully automated firewall rules + live updates<br>
                            <span class="check">✓</span> Auto plugin/core patching
                        </td>
                        <td>
                            <span class="cross">✗</span> No automation<br>
                            <span class="cross">✗</span> Manual updates required
                        </td>
                        <td>
                            <span class="warning">⚠</span> Scheduled scans only
                        </td>
                        <td>
                            <span class="warning">⚠</span> No automated mitigation
                        </td>
                    </tr>
                    
                    <!-- Backups -->
                    <tr>
                        <td class="feature-name">Backups</td>
                        <td class="highlight">
                            <span class="check">✓</span> Daily or hourly backups (up to 365-day retention)<br>
                            <span class="check">✓</span> Cloud-based, secure
                        </td>
                        <td>
                            <span class="warning">⚠</span> Manual or 3rd-party needed
                        </td>
                        <td>
                            <span class="check">✓</span> Included (short retention)
                        </td>
                        <td>
                            <span class="cross">✗</span> Not included, requires add-on
                        </td>
                    </tr>
                    
                    <!-- Firewall -->
                    <tr>
                        <td class="feature-name">Firewall</td>
                        <td class="highlight">
                            <span class="premium">🔒</span> AI-trained firewall<br>
                            <span class="check">✓</span> Blocks based on behavior & rules
                        </td>
                        <td>
                            <span class="warning">⚠</span> Basic WAF
                        </td>
                        <td>
                            <span class="check">✓</span> Basic WAF
                        </td>
                        <td>
                            <span class="warning">⚠</span> Basic cloud WAF
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
        
        <div class="wpfort-upgrade-cta">
            <a href="https://www.wpfort.ai/pricing" target="_blank" class="wpfort-btn wpfort-btn-primary wpfort-btn-large">Upgrade Your Protection</a>
        </div>
    </div>
    <?php endif; ?>
</div>
