<?php
/**
 * Simple Engine Debug Script
 * Tests individual engines with minimal WordPress dependencies
 */

echo esc_html("🔧 SIMPLE ENGINE DEBUG\n");
echo esc_html("======================\n\n");

// Set up basic WordPress constants
if (!defined('ABSPATH')) {
    define('ABSPATH', '/var/www/sub2.test-wpworld.uk/htdocs/');
}

// Test file path
$test_file = '/var/www/sub2.test-wpworld.uk/htdocs/wp-content/plugins/wpfortai-security/test-heuristic-malware.php';

echo esc_html("📁 Testing file: ") . esc_html($test_file) . esc_html("\n");
echo esc_html("📏 File exists: ") . esc_html(file_exists($test_file) ? 'YES' : 'NO') . esc_html("\n");

if (file_exists($test_file)) {
    echo esc_html("📏 File size: ") . esc_html(filesize($test_file)) . esc_html(" bytes\n");
    echo esc_html("📄 Content preview:\n");
    echo esc_html(substr(file_get_contents($test_file), 0, 200)) . esc_html("\n\n");
}

// Include engine files
$engine_path = '/var/www/sub2.test-wpworld.uk/htdocs/wp-content/plugins/wpfortai-security/includes/engines';

echo esc_html("🔧 Loading heuristic engine...\n");
if ($wp_filesystem->exists($engine_path . '/heuristic.php')) {
    require_once $engine_path . '/heuristic.php';
    echo esc_html("✅ Heuristic engine loaded\n");
} else {
    echo esc_html("❌ Heuristic engine file not found\n");
}

echo esc_html("🔧 Loading anomaly engine...\n");
if ($wp_filesystem->exists($engine_path . '/anomaly.php')) {
    require_once $engine_path . '/anomaly.php';
    echo esc_html("✅ Anomaly engine loaded\n");
} else {
    echo "❌ Anomaly engine not found\n";
}

// Mock WordPress functions that engines might need
if (!function_exists('get_option')) {
    function get_option($option, $default = false) {
        if ($option === 'wpsec_deep_scan_mode') return true;
        return $default;
    }
}

if (!function_exists('get_transient')) {
    function get_transient($key) { return false; }
}

if (!function_exists('set_transient')) {
    function set_transient($key, $value, $duration) { return true; }
}

if (!function_exists('delete_transient')) {
    function delete_transient($key) { return true; }
}

if (!defined('HOUR_IN_SECONDS')) define('HOUR_IN_SECONDS', 3600);
if (!defined('DAY_IN_SECONDS')) define('DAY_IN_SECONDS', 86400);

echo esc_html("\n🧪 TESTING HEURISTIC ENGINE:\n");
echo esc_html("============================\n");

if (function_exists('wpsec_check_file_heuristics')) {
    try {
        $heuristic_result = wpsec_check_file_heuristics($test_file, 'php');
        echo esc_html("✅ Heuristic function executed\n");
        echo esc_html("📊 Result type: ") . esc_html(gettype($heuristic_result)) . esc_html("\n");
        echo esc_html("📊 Result count: ") . esc_html(is_array($heuristic_result) ? count($heuristic_result) : 'N/A') . esc_html("\n");
        echo esc_html("📊 Result: ") . esc_html(json_encode($heuristic_result, JSON_PRETTY_PRINT)) . esc_html("\n");
    } catch (Exception $e) {
        echo esc_html("❌ Heuristic error: ") . esc_html($e->getMessage()) . esc_html("\n");
        echo esc_html("📍 Stack trace: ") . esc_html($e->getTraceAsString()) . esc_html("\n");
    }
} else {
    echo esc_html("❌ wpsec_check_file_heuristics function not found\n");
}

echo esc_html("\n🧪 TESTING ANOMALY ENGINE:\n");
echo esc_html("==========================\n");

if (function_exists('wpsec_check_file_anomalies')) {
    try {
        $anomaly_result = wpsec_check_file_anomalies($test_file, 'php');
        echo esc_html("✅ Anomaly function executed\n");
        echo esc_html("📊 Result type: ") . esc_html(gettype($anomaly_result)) . esc_html("\n");
        echo esc_html("📊 Result count: ") . esc_html(is_array($anomaly_result) ? count($anomaly_result) : 'N/A') . esc_html("\n");
        echo esc_html("📊 Result: ") . esc_html(json_encode($anomaly_result, JSON_PRETTY_PRINT)) . esc_html("\n");
    } catch (Exception $e) {
        echo esc_html("❌ Anomaly error: ") . esc_html($e->getMessage()) . esc_html("\n");
        echo esc_html("📍 Stack trace: ") . esc_html($e->getTraceAsString()) . esc_html("\n");
    }
} else {
    echo esc_html("❌ wpsec_check_file_anomalies function not found\n");
}

echo esc_html("\n🎯 DEBUG COMPLETE\n");
