<?php
defined('ABSPATH') || exit;

/**
 * Resume Scan Endpoint
 * 
 * This endpoint allows resuming a paused or stalled scan from a specific position.
 * It provides both REST API and plain permalink access options.
 * 
 * @param WP_REST_Request $request The request object
 * @return WP_REST_Response The response object
 */
function wpsec_resume_scan_endpoint($request) {
    // Validate API key
    if (!wpsec_validate_api_key()) {
        return new WP_Error(
            'rest_forbidden',
            'Sorry, you are not allowed to do that.',
            array('status' => 401)
        );
    }

    // Get parameters
    $scan_id = isset($request['scan_id']) ? sanitize_text_field($request['scan_id']) : '';
    $resume_from = isset($request['resume_from']) ? intval($request['resume_from']) : 0;
    $force_deep_scan = isset($request['force_deep_scan']) ? (bool)$request['force_deep_scan'] : false;
    $verbose_logging = isset($request['verbose_logging']) ? (bool)$request['verbose_logging'] : false;

    // Validate scan ID
    if (empty($scan_id)) {
        return rest_ensure_response([
            'status' => 'error',
            'message' => 'Missing required parameter: scan_id'
        ]);
    }

    // Check if scan exists
    $status = get_option('wpsec_scan_' . $scan_id . '_status', '');
    if (empty($status)) {
        return rest_ensure_response([
            'status' => 'error',
            'message' => 'Scan not found'
        ]);
    }

    // Check if scan is already completed
    if ($status === 'complete' || $status === 'completed') {
        return rest_ensure_response([
            'status' => 'error',
            'message' => 'Scan is already completed',
            'scan_id' => $scan_id,
            'scan_status' => $status
        ]);
    }

    // Load scan recovery module if not already loaded
    if (!function_exists('wpsec_resume_scan')) {
        require_once WPSEC_PLUGIN_DIR . 'includes/scan-recovery.php';
    }

    wpsec_debug_log('⚡️ WPFort: Manual scan resume triggered for scan ' . $scan_id . ' from position ' . $resume_from, 'info');

    // Store optional parameters if provided
    if ($force_deep_scan) {
        update_option('wpsec_scan_' . $scan_id . '_deep_scan', true);
        wpsec_debug_log('🔍 WPFort: Deep scan mode enabled for resumed scan', 'info');
    }

    if ($verbose_logging) {
        update_option('wpsec_scan_' . $scan_id . '_verbose_logging', true);
        wpsec_debug_log('📝 WPFort: Verbose logging enabled for resumed scan', 'info');
    }

    // Attempt to resume the scan
    $result = wpsec_resume_scan($scan_id, $resume_from);

    if ($result) {
        return rest_ensure_response([
            'status' => 'success',
            'message' => 'Scan resume initiated',
            'scan_id' => $scan_id,
            'resumed_from' => $resume_from,
            'deep_scan' => $force_deep_scan,
            'verbose_logging' => $verbose_logging
        ]);
    } else {
        return rest_ensure_response([
            'status' => 'error',
            'message' => 'Failed to resume scan',
            'scan_id' => $scan_id
        ]);
    }
}
