<?php
defined('ABSPATH') || exit;

function wpsec_scan_endpoint($request) {
    // Debug logging
    wpsec_debug_log('WPSEC Debug - Scan endpoint called', 'debug');
    
    // Validate API key (double check)
    if (!wpsec_validate_api_key()) {
        wpsec_debug_log('WPSEC Debug - API key validation failed', 'debug');
        return new WP_Error(
            'rest_forbidden',
            'Sorry, you are not allowed to do that.',
            array('status' => 401)
        );
    }
    
    // Check if webhooks should be bypassed
    $bypass_webhooks = isset($request['bypass_webhooks']) && ($request['bypass_webhooks'] === 'true' || $request['bypass_webhooks'] === true);
    wpsec_debug_log('WPSEC Debug - Bypass webhooks: ' . ($bypass_webhooks ? 'true' : 'false'), 'debug');
    
    // Check if deep scan was requested
    // Process the force_deep_scan parameter consistently across URL, GET, or POST
    $force_deep_scan = false; // Default to false, let's be explicit
    
    if (isset($request['force_deep_scan']) && 
        ($request['force_deep_scan'] === 'true' || $request['force_deep_scan'] === true || 
         $request['force_deep_scan'] === '1' || $request['force_deep_scan'] === 1)) {
        $force_deep_scan = true;
    } elseif (isset($_GET['force_deep_scan']) && 
             ($_GET['force_deep_scan'] === 'true' || $_GET['force_deep_scan'] === '1' || $_GET['force_deep_scan'] == 1)) {
        $force_deep_scan = true;
    }
    wpsec_debug_log(' WPSEC Debug - Force deep scan: ' . ($force_deep_scan ? 'YES' : 'NO'), 'debug');
    
    // Process verbose logging parameter
    $verbose_logging = false; // Default to false
    
    if (isset($request['verbose_logging']) && 
        ($request['verbose_logging'] === 'true' || $request['verbose_logging'] === true || 
         $request['verbose_logging'] === '1' || $request['verbose_logging'] === 1)) {
        $verbose_logging = true;
    } elseif (isset($_GET['verbose_logging']) && 
             ($_GET['verbose_logging'] === 'true' || $_GET['verbose_logging'] === '1' || $_GET['verbose_logging'] == 1)) {
        $verbose_logging = true;
    }
    wpsec_debug_log(' WPSEC Debug - Verbose logging: ' . ($verbose_logging ? 'ENABLED' : 'DISABLED'), 'debug');

    // Process test_files_only parameter for development/debugging
    $test_files_only = false; // Default to false
    
    if (isset($request['test_files_only']) && 
        ($request['test_files_only'] === 'true' || $request['test_files_only'] === true || 
         $request['test_files_only'] === '1' || $request['test_files_only'] === 1)) {
        $test_files_only = true;
        wpsec_debug_log('⚠️ TEST FILES MODE explicitly requested via request parameter', 'warning');
    } elseif (isset($_GET['test_files_only']) && 
             ($_GET['test_files_only'] === 'true' || $_GET['test_files_only'] === '1' || $_GET['test_files_only'] == 1)) {
        $test_files_only = true;
        wpsec_debug_log('⚠️ TEST FILES MODE explicitly requested via GET parameter', 'warning');
    }
    wpsec_debug_log(' WPSEC Debug - Test files only: ' . ($test_files_only ? 'ENABLED (dev mode)' : 'DISABLED (full scan)'), 'debug');

    wpsec_debug_log('WPSEC Debug - API key validation passed', 'debug');
    wpsec_debug_log('WPSEC Debug - Starting background scan process', 'debug');

    // Generate a unique scan ID
    $scan_id = uniqid('wpsec_', true);
    
    // Save initial scan metadata (minimal set to return response faster)
    update_option('wpsec_current_scan_id', $scan_id);
    update_option('wpsec_scan_' . $scan_id . '_start', time());
    update_option('wpsec_scan_' . $scan_id . '_status', 'initializing');
    update_option('wpsec_scan_' . $scan_id . '_files_scanned', 0);
    update_option('wpsec_scan_' . $scan_id . '_progress', 0);
    update_option('wpsec_scan_' . $scan_id . '_bypass_webhooks', $bypass_webhooks);
    update_option('wpsec_scan_' . $scan_id . '_force_deep_scan', $force_deep_scan);
    update_option('wpsec_scan_' . $scan_id . '_verbose_logging', $verbose_logging);
    
    // CRITICAL: Only save test_files_only as scan-specific option, NEVER as global option
    if ($test_files_only) {
        // Only set this if explicitly requested via parameter
        update_option('wpsec_scan_' . $scan_id . '_test_files_only', true);
        wpsec_debug_log('⚠️ TEST FILES MODE: Set scan-specific test_files_only=TRUE for scan_id=' . $scan_id, 'warning');
    } else {
        // For normal production scans, ensure test_files_only is explicitly FALSE
        update_option('wpsec_scan_' . $scan_id . '_test_files_only', false);
        wpsec_debug_log('✅ PRODUCTION SCAN: Set scan-specific test_files_only=FALSE for scan_id=' . $scan_id, 'info');
    }
    
    wpsec_debug_log(' Background scan initialization started at ' . current_time('mysql') . ' with ID: ' . $scan_id, 'info');
    
    // Prepare response with useful information
    $response = [
        'status' => 'success',
        'message' => 'Scan started and running in the background',
        'scan_id' => $scan_id,
        'started_at' => current_time('mysql'),
        'estimated_duration' => get_option('wpsec_last_scan_duration', '30.68') . ' seconds',
        'check_status_endpoint' => rest_url('wpsec/v1/scan-status/' . $scan_id),
        'results_endpoint' => rest_url('wpsec/v1/results'),
        'errors_endpoint' => rest_url('wpsec/v1/scan-errors/' . $scan_id)
    ];
    
    // Start the scan in the background
    ignore_user_abort(true);
    set_time_limit(0);

    if (function_exists('fastcgi_finish_request')) {
        // Make sure there's no previous output
        if (ob_get_level()) {
            ob_clean();
        }
        
        // Send the response immediately
        $json_response = wp_json_encode($response);
        header('Content-Type: application/json');
        header('Content-Length: ' . strlen($json_response));
        echo $json_response;
        @ob_end_flush();
        @flush();
        
        fastcgi_finish_request();
        
        // AFTER the response is sent, now build the scan queue
        wpsec_debug_log('WPSEC Debug - Building scan queue...', 'debug');
        
        // Set global variable for test_files_only mode before building queue
        global $wpsec_test_files_only;
        $wpsec_test_files_only = $test_files_only;
        wpsec_debug_log('🔧 ENDPOINT: Set global wpsec_test_files_only = ' . ($test_files_only ? 'TRUE' : 'FALSE'), 'info');
        
        $scan_info = wpsec_build_scan_queue();
        $total_files = $scan_info['total_files'];
        
        // Update scan metadata with queue info
        update_option('wpsec_scan_' . $scan_id . '_total_files', $total_files);
        update_option('wpsec_scan_' . $scan_id . '_status', 'running');
        
        wpsec_debug_log('WPSEC Debug - Queue built, starting scan with ' . $total_files . ' files', 'debug');
        
        // Log what we're about to do
        wpsec_debug_log(' WPSEC Debug - Running scan with scan_id=' . $scan_id . ', deep_scan=' . ($force_deep_scan ? 'YES' : 'NO') . ', verbose=' . ($verbose_logging ? 'YES' : 'NO') . ', test_files_only=' . ($test_files_only ? 'YES' : 'NO'), 'debug');
        
        // Now run the scan with all parameters
        wpsec_run_scan($scan_id, $force_deep_scan, $verbose_logging, $test_files_only);
    } else {
        // No FastCGI - still need to build queue for status info but do it quickly
        wpsec_debug_log('WPSEC Debug - No FastCGI available, running in synchronous mode', 'debug');
        
        // Schedule the scan to run via wp_cron as fallback
        wpsec_debug_log(' WPSEC Debug - Scheduling scan via cron with scan_id=' . $scan_id . ', deep_scan=' . ($force_deep_scan ? 'YES' : 'NO') . ', verbose=' . ($verbose_logging ? 'YES' : 'NO') . ', test_files_only=' . ($test_files_only ? 'YES' : 'NO'), 'debug');
        wp_schedule_single_event(time(), 'wpsec_scan_cron', array($scan_id, $force_deep_scan, $verbose_logging, $test_files_only));
        return rest_ensure_response($response);
    }
}
