<?php
defined('ABSPATH') || exit;

/**
 * Endpoint handler for WPFort Security Uptime monitoring API
 */
class WPSEC_Uptime_Endpoint {
    
    /**
     * API namespace
     * @var string
     */
    private $namespace = 'wpsec/v1';
    
    /**
     * Register routes for uptime monitoring
     */
    public function register_routes() {
        register_rest_route($this->namespace, '/uptime', [
            'methods'             => 'GET',
            'callback'            => [$this, 'get_uptime_status'],
            'permission_callback' => function () {
                return wpsec_validate_api_key();
            }
        ]);
        
        register_rest_route($this->namespace, '/uptime/config', [
            'methods'             => 'GET',
            'callback'            => [$this, 'get_uptime_config'],
            'permission_callback' => function () {
                return wpsec_validate_api_key();
            }
        ]);
        
        register_rest_route($this->namespace, '/uptime/config', [
            'methods'             => 'POST',
            'callback'            => [$this, 'update_uptime_config'],
            'permission_callback' => function () {
                return wpsec_validate_api_key();
            },
            'args' => [
                'tier' => [
                    'required' => false,
                    'validate_callback' => function($param) {
                        return is_numeric($param) && $param >= 1 && $param <= 4;
                    }
                ],
                'enabled' => [
                    'required' => false,
                    'validate_callback' => function($param) {
                        return is_bool($param);
                    }
                ]
            ]
        ]);
    }
    
    /**
     * Get uptime status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_uptime_status($request) {
        // Include uptime functionality
        if (!function_exists('wpsec_get_uptime_status')) {
            require_once WPSEC_PLUGIN_DIR . 'includes/uptime.php';
        }

        // Get uptime status data
        $status = wpsec_get_uptime_status();
        
        // Log uptime check
        wpsec_log_uptime_check($status);
        
        // Return formatted response
        return rest_ensure_response([
            'success' => true,
            'data' => $status
        ]);
    }
    
    /**
     * Get uptime monitoring configuration
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_uptime_config($request) {
        // Include uptime functionality
        if (!function_exists('wpsec_get_tier_level')) {
            require_once WPSEC_PLUGIN_DIR . 'includes/uptime.php';
        }
        
        $tier = wpsec_get_tier_level();
        $enabled = get_option('wpsec_uptime_monitoring_enabled', true);
        $interval = wpsec_get_monitor_interval($tier);
        $interval_text = wpsec_get_monitor_interval_text($tier);
        
        return rest_ensure_response([
            'success' => true,
            'data' => [
                'enabled' => $enabled,
                'tier' => $tier,
                'interval_seconds' => $interval,
                'interval_text' => $interval_text,
                'last_updated' => get_option('wpsec_uptime_config_updated', 0)
            ]
        ]);
    }
    
    /**
     * Update uptime monitoring configuration
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function update_uptime_config($request) {
        $updated = false;
        
        // Update tier if provided
        if ($request->has_param('tier')) {
            $tier = (int) $request->get_param('tier');
            update_option('wpsec_license_tier', $tier);
            $updated = true;
        }
        
        // Update enabled status if provided
        if ($request->has_param('enabled')) {
            $enabled = (bool) $request->get_param('enabled');
            update_option('wpsec_uptime_monitoring_enabled', $enabled);
            $updated = true;
        }
        
        // Update timestamp
        if ($updated) {
            update_option('wpsec_uptime_config_updated', time());
        }
        
        // Return updated config
        return $this->get_uptime_config($request);
    }
}

// Initialize endpoint
function wpsec_init_uptime_endpoint() {
    $endpoint = new WPSEC_Uptime_Endpoint();
    $endpoint->register_routes();
}
add_action('rest_api_init', 'wpsec_init_uptime_endpoint');
