<?php
defined('ABSPATH') || exit;

/**
 * WPFort Advanced WordPress Login Scanner
 * 
 * This engine scans WordPress login files for security issues.
 */

/**
 * Scan a WordPress login file for security issues
 * 
 * @param string $file_path Path to the file to scan
 * @param array $wp_login_definitions WordPress login security definitions
 * @return array|null Detection result or null if clean
 */
function wpsec_scan_login_file($file_path, $wp_login_definitions) {
    // Skip files that don't exist
    if (!file_exists($file_path)) {
        return null;
    }
    
    // Get filename for login file identification
    $filename = basename($file_path);
    
    // Only scan login-related files
    $login_files = ['wp-login.php', 'login.php', 'wp-signup.php', 'admin-ajax.php', 'admin-post.php'];
    if (!in_array($filename, $login_files)) {
        return null;
    }
    
    // Load file content
    $content = file_get_contents($file_path);
    if (!$content) {
        return null;
    }
    
    // Check each login pattern
    foreach ($wp_login_definitions as $pattern_id => $pattern_data) {
        $pattern = $pattern_data[1]; // The regex pattern is typically in this position
        
        if (!$pattern || !is_string($pattern)) {
            continue; // Skip invalid patterns
        }
        
        // Check if the pattern matches
        if (@preg_match($pattern, $content, $matches)) {
            if (empty($matches[0])) {
                continue; // Skip empty matches
            }
            
            // Login file tampering is high risk
            $threat_score = 5;
            
            // Map to standard detection format
            return wpsec_map_advanced_detection(
                $file_path, 
                'wp_login', 
                $pattern_id, 
                $pattern_data, 
                $matches[0], 
                $threat_score, 
                90 // High confidence for login file tampering
            );
        }
    }
    
    return null;
}

/**
 * Scan WordPress login files for security issues
 * 
 * @param array $scan_queue File scan queue
 * @param array $wp_login_definitions WordPress login security definitions
 * @return array Array of detection results
 */
function wpsec_scan_login_files($scan_queue, $wp_login_definitions) {
    $results = [];
    
    if (!$wp_login_definitions || !is_array($wp_login_definitions)) {
        wpsec_debug_log('❌ WPFort: No WordPress login definitions available for scanning', 'error');
        return $results;
    }
    
    wpsec_debug_log('🔍 WPFort: Scanning WordPress login files with ' . count($wp_login_definitions) . ' definitions', 'info');
    
    $login_files = [];
    $login_filenames = ['wp-login.php', 'login.php', 'wp-signup.php', 'admin-ajax.php', 'admin-post.php'];
    
    // First, filter only login files from the scan queue for efficiency
    foreach ($scan_queue as $file_path => $priority) {
        $filename = basename($file_path);
        if (in_array($filename, $login_filenames)) {
            $login_files[$file_path] = $priority;
        }
    }
    
    wpsec_debug_log('🔍 WPFort: Found ' . count($login_files) . ' WordPress login files to scan', 'info');
    
    $matched = 0;
    
    // Now scan each login file
    foreach ($login_files as $file_path => $priority) {
        $result = wpsec_scan_login_file($file_path, $wp_login_definitions);
        
        if ($result) {
            $results[] = $result;
            $matched++;
        }
    }
    
    wpsec_debug_log(sprintf(
        '✅ WPFort: WordPress login scan complete - %d files scanned, %d matches found',
        count($login_files),
        $matched
    ), 'info');
    
    return $results;
}
