<?php
defined('ABSPATH') || exit;

/**
 * Detect server environment to optimize scan parameters
 * This function analyzes the current server environment to determine its capabilities
 * and constraints for better scan optimization
 * 
 * @return array Server environment information
 */
function wpsec_detect_server_environment() {
    // Get CPU load if available
    $cpu_load = 0;
    if (function_exists('sys_getloadavg')) {
        $load = sys_getloadavg();
        $cpu_load = isset($load[0]) ? min(100, round($load[0] * 100 / 4)) : 0; // Assume 4 cores as default
    }
    
    // Get memory information
    $memory_limit = ini_get('memory_limit');
    $memory_usage = memory_get_usage(true);
    $memory_limit_bytes = wpsec_get_memory_limit_bytes();
    $memory_usage_percent = round(($memory_usage / $memory_limit_bytes) * 100, 2);
    
    // Get disk space
    $disk_free_space = function_exists('disk_free_space') ? disk_free_space(ABSPATH) : 0;
    $disk_total_space = function_exists('disk_total_space') ? disk_total_space(ABSPATH) : 0;
    $disk_usage_percent = $disk_total_space > 0 ? round((($disk_total_space - $disk_free_space) / $disk_total_space) * 100, 2) : 0;
    
    // Check for shared hosting indicators
    $is_shared_hosting = false;
    
    // Low memory limit is a good indicator of shared hosting
    if ($memory_limit_bytes < 256 * 1024 * 1024) { // Less than 256MB
        $is_shared_hosting = true;
    }
    
    // Check for execution time limit
    $max_execution_time = ini_get('max_execution_time');
    if ($max_execution_time > 0 && $max_execution_time < 60) { // Less than 60 seconds
        $is_shared_hosting = true;
    }
    
    // Determine environment type based on resources
    $type = 'normal';
    
    if ($is_shared_hosting || $cpu_load > 70 || $memory_usage_percent > 70) {
        $type = 'constrained';
    }
    
    if ($cpu_load > 90 || $memory_usage_percent > 90) {
        $type = 'overloaded';
    }
    
    return [
        'type' => $type,
        'cpu_load' => $cpu_load,
        'memory_limit' => $memory_limit,
        'memory_available' => size_format($memory_limit_bytes - $memory_usage),
        'memory_usage_percent' => $memory_usage_percent,
        'disk_free_space' => size_format($disk_free_space),
        'max_execution_time' => $max_execution_time,
        'is_shared_hosting' => $is_shared_hosting
    ];
}

/**
 * Determine optimal chunk size based on server environment
 * 
 * @param array $environment Server environment information
 * @return int Optimal chunk size
 */
function wpsec_get_optimal_chunk_size($environment) {
    // Define a smaller default chunk size for better performance on high-CPU servers
    // Using a smaller but more efficient chunk size helps the scan complete more consistently
    $base_chunk_size = 750; // Reduced from 2000
    
    // Even on high CPU servers, we want to make progress, so don't make chunks too small
    // A balanced approach: smaller chunks but still large enough to make progress
    if ($environment['type'] === 'overloaded') {
        // Keep minimum size even when overloaded to maintain forward progress
        $base_chunk_size = 600;
    } else if ($environment['cpu_load'] > 90) {
        // Still process a good number of files even under high load
        $base_chunk_size = 700;
    }
    
    // Never go below 500 files per chunk to ensure we make progress
    return max(500, (int) $base_chunk_size);
}
