<?php
defined('ABSPATH') || exit;

/**
 * Reset a scan completely, removing all checkpoints and data
 * 
 * @param string $scan_id The scan ID to reset
 * @return bool Success
 */
function wpsec_reset_scan($scan_id) {
    if (empty($scan_id)) {
        wpsec_debug_log('⚠️ WPFort Reset: Empty scan ID provided', 'warning');
        return false;
    }
    
    // Validate scan ID format
    if (!preg_match('/^wpsec_[a-f0-9]+\.[0-9]+$/', $scan_id)) {
        wpsec_debug_log('⚠️ WPFort Reset: Invalid scan ID format: ' . $scan_id, 'warning');
        return false;
    }
    
    wpsec_debug_log('🔄 WPFort Reset: Resetting scan ' . $scan_id, 'info');
    
    // Stop any scheduled tasks for this scan
    $next_event = wp_next_scheduled('wpsec_run_resumable_scan', array($scan_id));
    if ($next_event) {
        wp_unschedule_event($next_event, 'wpsec_run_resumable_scan', array($scan_id));
        wpsec_debug_log('🔄 WPFort Reset: Unscheduled pending scan task', 'info');
    }
    
    // Get all checkpoint IDs
    $active_checkpoint = get_option('wpsec_scan_' . $scan_id . '_active_checkpoint', '');
    
    // Get list of option names to delete
    $options_to_delete = array(
        'wpsec_scan_' . $scan_id . '_status',
        'wpsec_scan_' . $scan_id . '_start',
        'wpsec_scan_' . $scan_id . '_end',
        'wpsec_scan_' . $scan_id . '_progress',
        'wpsec_scan_' . $scan_id . '_files_scanned',
        'wpsec_scan_' . $scan_id . '_files_processed', // Our simplified counter
        'wpsec_scan_' . $scan_id . '_processed_count',
        'wpsec_scan_' . $scan_id . '_total_files',
        'wpsec_scan_' . $scan_id . '_active_checkpoint',
        'wpsec_scan_' . $scan_id . '_last_checkpoint_time',
        'wpsec_scan_' . $scan_id . '_last_scanned_file',
        'wpsec_scan_' . $scan_id . '_last_progress_update',
        'wpsec_scan_' . $scan_id . '_file_errors',
        'wpsec_scan_' . $scan_id . '_files_cached'
    );
    
    // Delete options
    foreach ($options_to_delete as $option_name) {
        delete_option($option_name);
    }
    
    // Delete checkpoint options
    if (!empty($active_checkpoint)) {
        $checkpoint_options = array(
            'wpsec_scan_' . $scan_id . '_checkpoint_' . $active_checkpoint,
            'wpsec_scan_' . $scan_id . '_checkpoint_' . $active_checkpoint . '_queue',
            'wpsec_scan_' . $scan_id . '_checkpoint_' . $active_checkpoint . '_results',
            'wpsec_scan_' . $scan_id . '_checkpoint_' . $active_checkpoint . '_processed'
        );
        
        foreach ($checkpoint_options as $option_name) {
            delete_option($option_name);
        }
    }
    
    // Get all options to find any checkpoint data we missed
    global $wpdb;
    $like_pattern = 'wpsec_scan_' . $scan_id . '_checkpoint_%';
    $other_checkpoint_options = $wpdb->get_col(
        $wpdb->prepare(
            "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
            $like_pattern
        )
    );
    
    // Delete all found checkpoint options
    foreach ($other_checkpoint_options as $option_name) {
        delete_option($option_name);
    }
    
    wpsec_debug_log('✅ WPFort Reset: Successfully reset scan ' . $scan_id, 'info');
    
    // Send webhook notification about the reset if needed
    if (function_exists('wpsec_send_scan_webhook')) {
        wpsec_send_scan_webhook($scan_id, 'reset', array(
            'message' => 'Scan was manually reset'
        ));
    }
    
    return true;
}

/**
 * Reset a stalled scan automatically
 * 
 * @param string $scan_id The scan ID to check and reset if stalled
 * @param int $stall_threshold Time in seconds after which a scan is considered stalled
 * @return bool Whether the scan was reset
 */
function wpsec_check_and_reset_stalled_scan($scan_id, $stall_threshold = 300) {
    if (empty($scan_id)) {
        return false;
    }
    
    $last_update = get_option('wpsec_scan_' . $scan_id . '_last_progress_update', 0);
    $status = get_option('wpsec_scan_' . $scan_id . '_status', '');
    
    // Only reset running scans that have stalled
    if ($status !== 'running') {
        return false;
    }
    
    $time_since_update = time() - $last_update;
    
    if ($last_update > 0 && $time_since_update > $stall_threshold) {
        wpsec_debug_log('⚠️ WPFort: Scan ' . $scan_id . ' appears to be stalled (' . $time_since_update . ' seconds since last update). Auto-resetting.', 'warning');
        
        // Mark as failed before resetting
        update_option('wpsec_scan_' . $scan_id . '_status', 'failed');
        
        // Send webhook notification about the failure
        if (function_exists('wpsec_send_scan_webhook')) {
            wpsec_send_scan_webhook($scan_id, 'failed', array(
                'reason' => 'Scan stalled for ' . $time_since_update . ' seconds'
            ));
        }
        
        // Reset the scan
        return wpsec_reset_scan($scan_id);
    }
    
    return false;
}
